import os

import hydra
from omegaconf import DictConfig, OmegaConf
import subprocess
from run_experiment import run_seed
import yaml 

@hydra.main(version_base=None, config_path="conf", config_name="config")
def run(cfg: DictConfig) -> None:
    """Run the experiment with the given configuration. In a multirun
    environment, this code will be executed for each run.

    Args:
        cfg (DictConfig): Configuration dictionary.
    """

    print(OmegaConf.to_yaml(cfg))
    print(f"Working directory : {os.getcwd()}")

    # Set output directory
    output_dir = hydra.core.hydra_config.HydraConfig.get().runtime.output_dir
    print(f"Output directory  : {output_dir}")

    # The list of seeds comes as a list of strings, we need to convert it to a list of integers.
    seed_list = [int(seed) for seed in cfg["seeds_list"]]

    # If the model is deterministic, the random seed has no effect, so we only
    # run the experiment for the first seed in the list.
    if cfg.experiment.deterministic:
        seed_list = [seed_list[0]]

    seed_path_list = []

    for seed in seed_list:
        print(f"Running experiment for seed: {seed}")

        seed_path = os.path.join(output_dir, f"seed_{seed}")
        seed_path_list.append(seed_path)
        os.makedirs(seed_path, exist_ok=True)

        cfg_path = os.path.join(output_dir, ".hydra", "config.yaml")

        if not cfg.sequential:
            script_path = os.path.join(
                os.getcwd(), "src", "run_experiment.py"
            )  # Run this script from the main directory, not from the src directory
            command = (
                f"python {script_path} "
                + f"--config_path {cfg_path} --experiment_dir {seed_path} "
                + f" --seed {seed}"
            )
            subprocess.run(command)
        else:
            # Load the config file and run the experiment
            with open(cfg_path, "r") as f:
                config = OmegaConf.create(yaml.safe_load(f))

            results, runtime = run_seed(config, seed_path, seed)

            results.to_csv(os.path.join(seed_path, f"results_RS{seed}.csv"), index=False)
            runtime.to_csv(os.path.join(seed_path, f"runtime_RS{seed}.csv"), index=False)
            print(f"Seed {seed} completed successfully.")



    return


if __name__ == "__main__":
    run()
